<?php
/* --------------------------------------------------------------
   GambioHubCheckoutConfirmationContentControl.inc.php 2023-07-17
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2023 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

/**
 * Class GambioHubCheckoutConfirmationContentControl
 */
class GambioHubCheckoutConfirmationContentControl extends GambioHubCheckoutConfirmationContentControl_parent
{
    /**
     * Maximum age of a Gambio Hub session in seconds
     */
    const SESSION_MAX_AGE = 3300;
    public function proceed()
    {
        parent::proceed();
        $this->setRedirectToCheckoutProcessIfNeeded();
    }

    /**
     * @param $error
     *
     * @return boolean
     */
    protected function _checkTransportConditions(&$error)
    {
    	// set gambio hub module selection before redirecting to checkout payment page in error cases
	    if(isset($_POST) && array_key_exists('gambio_hub_selection', $_POST))
	    {
		    $_SESSION['gambio_hub_selection'] = $_POST['gambio_hub_selection'];
	    }
    	
        if(array_key_exists('shipping', $_SESSION) && is_array($_SESSION['shipping'])
           && array_key_exists('id', $_SESSION['shipping']) && !is_array(explode(',', gm_get_conf('DATA_TRANSFER_TO_TRANSPORT_COMPANIES_SETTINGS')))
        )
        {
            return true;
        }
        
        return parent::_checkTransportConditions($error);
    }
    
    
    /**
     * Causes redirect back to payment selection in case Hub session is too old.
     *
     * @return bool
     */
    public function check_payment()
    {
        $paymentAllowed = parent::check_payment();
    
        if (isset($this->v_data_array['POST']['payment'])
            && preg_match('/gambio_hub-(?<hubselection>[\w]+)(-(?<hubsubselection>[^\s]+))?/',
                $this->v_data_array['POST']['payment'], $matches) === 1) {
            $this->v_data_array['POST']['payment'] = 'gambio_hub';
            $_POST['payment'] = 'gambio_hub';
            $this->v_data_array['POST']['gambio_hub_selection'] = $matches['hubselection'];
            $_POST['gambio_hub_selection'] = $matches['hubselection'];
            if (!empty($matches['hubsubselection'])) {
                $this->v_data_array['POST']['gambio_hub_subselection'] = $matches['hubsubselection'];
                $_POST['gambio_hub_subselection'] = $matches['hubsubselection'];
            } else {
                $this->v_data_array['POST']['gambio_hub_subselection'] = '';
                $_POST['gambio_hub_subselection'] = '';
            }
        }
        
        if (($_SESSION['payment'] ?? null) === 'gambio_hub') {
            $sessionTooOld = empty($_SESSION['gambio_hub_session_key_refreshed'])
                             || (microtime(true) - $_SESSION['gambio_hub_session_key_refreshed'])
                                > self::SESSION_MAX_AGE;
            if ($sessionTooOld) {
                $paymentAllowed = false;
            }
        }
        
        return $paymentAllowed;
    }

    /**
     * Redirect to CheckoutProcess if merchant enable PayPal PayNow
     *
     * @return void
     */
    private function setRedirectToCheckoutProcessIfNeeded(): void
    {
        try {
            if ($this->isMerchantEnablePayNow() === false) {
                return;
            }
            if ($this->isPayPal2HubEcmSelected() && empty($this->get_redirect_url())) {
                $this->set_redirect_url(xtc_href_link(FILENAME_CHECKOUT_PROCESS));
            }
        } catch (Exception $e) {
            return;
        }
    }
    
    /**
     * PayNow only apply for PayPal ecm 
     * PayPal Express checkout (ecs) and another payment method will remain
     * 
     * @return bool
     */
    private function isPayPal2HubEcmSelected(): bool
    {
        $post = $this->v_data_array['POST'] ?? [];

        $paymentMethod = $post['payment'] ?? null;
        $initiator = $post['PayPal2HubPaymentInitiator'] ?? null;
        $hubSelection = $post['gambio_hub_selection'] ?? null;
        $hubSubselection = $post['gambio_hub_subselection'] ?? null;
        
        // Check the main PayPal2Hub ECM conditions
        if ($paymentMethod !== 'gambio_hub') return false;
        if ($initiator !== 'ecm') return false;
        if ($hubSelection !== 'PayPal2Hub') return false;
        
        return $this->isValidHubSubselection($hubSubselection);
    }

    /**
     * @param string|null $subselection
     * @return bool
     */
    private function isValidHubSubselection(?string $subselection): bool
    {
        // Empty or one of the allowed digital wallets
        $allowedSubselections = ['applepay', 'googlepay'];
        return empty($subselection) || in_array($subselection, $allowedSubselections, true);
    }
    
    /**
     * Check merchant enable pay now 
     * @return bool
     */
    private function isMerchantEnablePayNow() : bool
    {
        $config = MainFactory::create('HubPayPalConfiguration');
        $paypalSettings = MainFactory::create('HubPayPalButtonSettings', $config);
        $merchantSettings = $paypalSettings->getButtonSettings();
        
        return $merchantSettings['usePayNow'] === true;
    }
}
